#include <SDI12.h>

#define SERIAL_BAUD 115200  // The baud rate for the output serial port
#define DATA_PIN 7         // The pin of the SDI-12 data bus
#define POWER_PIN 22       // The sensor power pin (or -1 if not switching power)


// Define the SDI-12 bus
SDI12 mySDI12(DATA_PIN);

String myCommand = "";   // empty to start
char oldAddress = '!';   // invalid address as placeholder

// This checks for activity at a particular address
boolean checkActive(byte i) {
  Serial.print("Checking address: ");
  Serial.print((char)i);
  Serial.print("...");

  myCommand = "";
  myCommand += (char) i;  // sends basic 'acknowledge' command [address][!]
  myCommand += "!";

  for (int j = 0; j < 3; j++) {  // goes through three rapid contact attempts
    Serial.print("Attempt #");
    Serial.println(j + 1);

    mySDI12.sendCommand(myCommand);
    delay(3000);
    
    if (mySDI12.available()) {  // If we hear anything, assume we have an active sensor
      Serial.println("Occupied");
      mySDI12.clearBuffer();
      return true;
    } else {
      Serial.println("Vacant");
      mySDI12.clearBuffer();
    }
  }
  return false;
}

void setup() {
  Serial.begin(SERIAL_BAUD);
  while (!Serial);

  Serial.println("Opening SDI-12 bus...");
  mySDI12.begin();
  delay(500);  // Allow things to settle

  // Power the sensors
  if (POWER_PIN > 0) {
    Serial.println("Powering up sensors...");
    pinMode(POWER_PIN, OUTPUT);
    digitalWrite(POWER_PIN, HIGH);
    delay(3000);  // Increase the delay to 3000ms (3 seconds) to ensure proper power-up time
    Serial.println("Sensor powered up.");
  }

  // Checking if the bus is working and the sensor is responding
  Serial.println("Checking if sensor is responding...");
  boolean sensorFound = false;

  for (byte i = '0'; i <= '9'; i++) {  // scan address space 0-9
    if (checkActive(i)) {
      sensorFound = true;
      oldAddress = i;
      break;
    }
  }

  if (!sensorFound) {
    for (byte i = 'a'; i <= 'z'; i++) {  // scan address space a-z
      if (checkActive(i)) {
        sensorFound = true;
        oldAddress = i;
        break;
      }
    }
  }

  if (!sensorFound) {
    for (byte i = 'A'; i <= 'Z'; i++) {  // scan address space A-Z
      if (checkActive(i)) {
        sensorFound = true;
        oldAddress = i;
        break;
      }
    }
  }

  if (!sensorFound) {
    Serial.println("No sensor detected. Please check connections or sensor address.");
  } else {
    Serial.print("Sensor active at address: ");
    Serial.println(oldAddress);
  }
}

void loop() {
  // This part of the code can stay mostly unchanged for readdressing purposes
  if (oldAddress != '!') {
    Serial.println("Enter new address.");
    while (!Serial.available());
    char newAdd = Serial.read();

    // Wait for valid response
    while (((newAdd < '0') || (newAdd > '9')) && ((newAdd < 'a') || (newAdd > 'z')) && ((newAdd < 'A') || (newAdd > 'Z'))) {
      if (!(newAdd == '\n') || (newAdd == '\r') || (newAdd == ' ')) {
        Serial.println("Not a valid address. Please enter '0'-'9', 'a'-'A', or 'z'-'Z'.");
      }
      while (!Serial.available());
      newAdd = Serial.read();
    }

    // Syntax for changing address: [currentAddress]A[newAddress]!
    Serial.println("Readdressing sensor...");
    myCommand = "";
    myCommand += (char) oldAddress;
    myCommand += "A";
    myCommand += (char) newAdd;
    myCommand += "!";
    mySDI12.sendCommand(myCommand);

    // Wait for the response and clear buffer
    delay(300);
    mySDI12.clearBuffer();

    Serial.println("Success. Rescanning for verification.");
  }
}
